/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { AmountString, TalerCorebankApiClient } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

/**
 * Run test for refreshe after a payment.
 */
export async function runWalletRefreshErrorsTest(t: GlobalTestState) {
  // Set up test environment

  const coinConfigList: CoinConfig[] = [
    {
      ...coinCommon,
      name: "n1",
      value: "TESTKUDOS:1",
    },
    {
      ...coinCommon,
      name: "n5",
      value: "TESTKUDOS:5",
    },
  ];

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t, coinConfigList);

  const bankClient = new TalerCorebankApiClient(bank.baseUrl);

  const wres = await withdrawViaBankV3(t, {
    amount: "TESTKUDOS:5",
    bankClient,
    exchange,
    walletClient,
  });
  await wres.withdrawalFinishedCond;

  const backupResp = await walletClient.call(
    WalletApiOperation.CreateStoredBackup,
    {},
  );

  const coinDump = await walletClient.call(WalletApiOperation.DumpCoins, {});

  t.assertDeepEqual(coinDump.coins.length, 1);

  await walletClient.call(WalletApiOperation.ForceRefresh, {
    refreshCoinSpecs: [
      {
        coinPub: coinDump.coins[0].coinPub,
        amount: "TESTKUDOS:3" as AmountString,
      },
    ],
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  await walletClient.call(WalletApiOperation.RecoverStoredBackup, {
    name: backupResp.name,
  });

  await walletClient.call(WalletApiOperation.ForceRefresh, {
    refreshCoinSpecs: [
      {
        coinPub: coinDump.coins[0].coinPub,
        amount: "TESTKUDOS:3" as AmountString,
      },
    ],
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Now reset everything and do more tests

  await walletClient.call(WalletApiOperation.ClearDb, {});
  {
    const wres = await withdrawViaBankV3(t, {
      amount: "TESTKUDOS:5",
      bankClient,
      exchange,
      walletClient,
    });
    await wres.withdrawalFinishedCond;

    const coinDump = await walletClient.call(WalletApiOperation.DumpCoins, {});

    t.assertDeepEqual(coinDump.coins[0].denomValue, "TESTKUDOS:5");

    await walletClient.call(WalletApiOperation.ForceRefresh, {
      refreshCoinSpecs: [
        {
          coinPub: coinDump.coins[0].coinPub,
          amount: "TESTKUDOS:5" as AmountString,
        },
      ],
    });
    // Do it twice!
    await walletClient.call(WalletApiOperation.ForceRefresh, {
      refreshCoinSpecs: [
        {
          coinPub: coinDump.coins[0].coinPub,
          amount: "TESTKUDOS:5" as AmountString,
        },
      ],
    });

    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
  }
}

runWalletRefreshErrorsTest.suites = ["wallet"];
