/*
   This file is part of GNU Taler
   (C) 2021-2025 Taler Systems S.A.

   GNU Taler is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
   */

/**
 *
 * @author Martin Schanzenbach
 */

import {
  AbsoluteTime,
  Duration,
  StatisticBucketRange
} from "@gnu-taler/taler-util";
import { VNode, h } from "preact";
import { useState } from "preact/hooks";

import "chartjs-adapter-date-fns";
import { useSessionContext } from "../../../../context/session.js";
import {
  MerchantOrderStatsSlug,
  MerchantRevenueStatsSlug
} from "../../../../hooks/statistics.js";
import { OrdersChart } from "./OrdersChart.js";
import { RevenueChart } from "./RevenueChart.js";
interface Props {}


export interface StatSlug {
  slug: string;
  text: string;
}

const chartColors = new Map<
  MerchantOrderStatsSlug | MerchantRevenueStatsSlug,
  string
>();
chartColors.set("orders-created", "#b9a5ff");
// chartColors.set("tokens-issued", "#b9a5ff");
chartColors.set("payments-received-after-deposit-fee", "#b9a5ff");
chartColors.set("orders-claimed", "#647cda");
chartColors.set("total-wire-fees-paid", "#647cda");
chartColors.set("total-deposit-fees-paid", "#2830a8");
chartColors.set("orders-settled", "#2830a8");
// chartColors.set("tokens-used", "#2830a8");
chartColors.set("orders-paid", "#525597");
chartColors.set("refunds-granted", "#525597");

export interface RevenueChartFilter {
  rangeCount: number;
  range: RelevantTimeUnit;
  currency: string;
}

export type RelevantTimeUnit =
  | "hour"
  | "day"
  | "week"
  | "month"
  | "quarter"
  | "year";

export default function Statistics({}: Props): VNode {
  const { config } = useSessionContext();
  const lastMonthAbs = AbsoluteTime.subtractDuraction(
    AbsoluteTime.now(),
    Duration.fromSpec({ months: 1 }),
  );

  const [revenueChartFilter, setRevenueChartFilter] =
    useState<RevenueChartFilter>({
      range: StatisticBucketRange.Quarter,
      rangeCount: 4,
      currency: config.currency,
    });
  const [startOrdersFromDate, setStartOrdersFromDate] = useState<
    AbsoluteTime | undefined
  >(lastMonthAbs);

  const availableCurrencies = Object.keys(config.currencies);
  return (
    <section class="section is-main-section">
      <div>
        <RevenueChart
          colors={chartColors as Map<MerchantRevenueStatsSlug, string>}
          activeFilter={revenueChartFilter}
          availableCurrencies={availableCurrencies}
          onUpdateFilter={(filter: RevenueChartFilter) =>
            setRevenueChartFilter(filter)
          }
        />
      </div>
      <hr />
      <OrdersChart
        colors={chartColors as Map<MerchantOrderStatsSlug, string>}
        filterFromDate={startOrdersFromDate}
        onSelectDate={(d?: AbsoluteTime) => {
          setStartOrdersFromDate(d);
        }}
      />
    </section>
  );
}
