/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_delete_unit.c
 * @brief command to test DELETE /private/units/$ID
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State for a DELETE /private/units/$ID command.
 */
struct DeleteUnitState
{
  /**
   * In-flight request handle.
   */
  struct TALER_MERCHANT_UnitDeleteHandle *udh;

  /**
   * Interpreter context.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Merchant backend base URL.
   */
  const char *merchant_url;

  /**
   * Unit identifier to delete.
   */
  const char *unit_id;

  /**
   * Expected HTTP status.
   */
  unsigned int http_status;
};


/**
 * Completion callback.
 */
static void
delete_unit_cb (void *cls,
                const struct TALER_MERCHANT_HttpResponse *hr)
{
  struct DeleteUnitState *dus = cls;

  dus->udh = NULL;
  if (dus->http_status != hr->http_status)
  {
    TALER_TESTING_unexpected_status_with_body (dus->is,
                                               hr->http_status,
                                               dus->http_status,
                                               hr->reply);
    return;
  }
  TALER_TESTING_interpreter_next (dus->is);
}


/**
 * Issue DELETE request.
 */
static void
delete_unit_run (void *cls,
                 const struct TALER_TESTING_Command *cmd,
                 struct TALER_TESTING_Interpreter *is)
{
  struct DeleteUnitState *dus = cls;

  dus->is = is;
  dus->udh = TALER_MERCHANT_unit_delete (
    TALER_TESTING_interpreter_get_context (is),
    dus->merchant_url,
    dus->unit_id,
    &delete_unit_cb,
    dus);
  if (NULL == dus->udh)
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
  }
}


/**
 * Provide traits.
 */
static enum GNUNET_GenericReturnValue
delete_unit_traits (void *cls,
                    const void **ret,
                    const char *trait,
                    unsigned int index)
{
  struct DeleteUnitState *dus = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_make_trait_unit_id (dus->unit_id),
    TALER_TESTING_trait_end ()
  };

  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


/**
 * Cleanup.
 */
static void
delete_unit_cleanup (void *cls,
                     const struct TALER_TESTING_Command *cmd)
{
  struct DeleteUnitState *dus = cls;

  if (NULL != dus->udh)
  {
    TALER_MERCHANT_unit_delete_cancel (dus->udh);
    dus->udh = NULL;
  }
  GNUNET_free (dus);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_delete_unit (const char *label,
                                        const char *merchant_url,
                                        const char *unit_id,
                                        unsigned int http_status)
{
  struct DeleteUnitState *dus;

  dus = GNUNET_new (struct DeleteUnitState);
  dus->merchant_url = merchant_url;
  dus->unit_id = unit_id;
  dus->http_status = http_status;

  {
    struct TALER_TESTING_Command cmd = {
      .cls = dus,
      .label = label,
      .run = &delete_unit_run,
      .cleanup = &delete_unit_cleanup,
      .traits = &delete_unit_traits
    };

    return cmd;
  }
}


/* end of testing_api_cmd_delete_unit.c */
