/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file lib/exchange_api_aml_legitimizations_get.c
 * @brief Implementation of the GET /aml/$OFFICER_PUB/legitimizations requests
 * @author Christian Grothoff
 */
#include "taler/platform.h"
#include <jansson.h>
#include <microhttpd.h> /* just for HTTP status codes */
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_json_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler/taler_exchange_service.h"
#include "taler/taler_json_lib.h"
#include "exchange_api_handle.h"
#include "taler/taler_signatures.h"
#include "exchange_api_curl_defaults.h"


/**
 * Handle for an operation to GET /aml/$OFFICER_PUB/legitimizations.
 */
struct TALER_EXCHANGE_AmlLegitimizationsGetHandle
{

  /**
   * The exchange base URL for this request.
   */
  char *exchange_base_url;

  /**
   * Our execution context.
   */
  struct GNUNET_CURL_Context *ctx;

  /**
   * Handle for the request.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Signature of the AML officer.
   */
  struct TALER_AmlOfficerSignatureP officer_sig;

  /**
   * Public key of the AML officer.
   */
  struct TALER_AmlOfficerPublicKeyP officer_pub;

  /**
   * Function to call with the result.
   */
  TALER_EXCHANGE_AmlLegitimizationsGetCallback cb;

  /**
   * Closure for @a cb.
   */
  TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_RESULT_CLOSURE *cb_cls;

  /**
   * The url for this request.
   */
  char *url;

  /**
   * HTTP headers for the job.
   */
  struct curl_slist *job_headers;

  /**
   * Request options.
   */
  struct
  {
    /**
     * Limit on number of results.
     */
    int64_t limit;

    /**
     * Row offset from which to return results.
     */
    uint64_t offset;

    /**
     * Hash of payto URI to filter by, NULL for no filter.
     */
    const struct TALER_NormalizedPaytoHashP *h_payto;

    /**
     * Activity filter.
     */
    enum TALER_EXCHANGE_YesNoAll active;

  } options;

};


/**
 * Parse a single measure details entry from JSON.
 *
 * @param md_json JSON object to parse
 * @param[out] md where to store the result
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
parse_measure_details (
  const json_t *md_json,
  struct TALER_EXCHANGE_AmlLegitimizationsGetMeasureDetails *md)
{
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_fixed_auto ("h_payto",
                                 &md->h_payto),
    GNUNET_JSON_spec_uint64 ("rowid",
                             &md->rowid),
    GNUNET_JSON_spec_timestamp ("start_time",
                                &md->start_time),
    GNUNET_JSON_spec_object_const ("measures",
                                   &md->measures),
    GNUNET_JSON_spec_bool ("is_finished",
                           &md->is_finished),
    GNUNET_JSON_spec_end ()
  };

  if (GNUNET_OK !=
      GNUNET_JSON_parse (md_json,
                         spec,
                         NULL,
                         NULL))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }
  return GNUNET_OK;
}


/**
 * We received an #MHD_HTTP_OK status code. Handle the JSON
 * response.
 *
 * @param algh handle of the request
 * @param j JSON response
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
handle_aml_legitimizations_get_ok (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh,
  const json_t *j)
{
  struct TALER_EXCHANGE_AmlLegitimizationsGetResult result = {
    .hr.reply = j,
    .hr.http_status = MHD_HTTP_OK
  };
  const json_t *measures_array;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_array_const ("measures",
                                  &measures_array),
    GNUNET_JSON_spec_end ()
  };
  struct TALER_EXCHANGE_AmlLegitimizationsGetMeasureDetails *measures;

  if (GNUNET_OK !=
      GNUNET_JSON_parse (j,
                         spec,
                         NULL,
                         NULL))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  result.details.ok.measures_length = json_array_size (measures_array);
  if (0 == result.details.ok.measures_length)
  {
    measures = NULL;
  }
  else
  {
    measures
      = GNUNET_new_array (
          result.details.ok.measures_length,
          struct TALER_EXCHANGE_AmlLegitimizationsGetMeasureDetails);
  }
  for (size_t i = 0; i < result.details.ok.measures_length; i++)
  {
    const json_t *measure_json = json_array_get (measures_array,
                                                 i);

    if (GNUNET_OK !=
        parse_measure_details (measure_json,
                               &measures[i]))
    {
      GNUNET_free (measures);
      return GNUNET_SYSERR;
    }
  }
  result.details.ok.measures = measures;
  algh->cb (algh->cb_cls,
            &result);
  algh->cb = NULL;
  GNUNET_free (measures);
  return GNUNET_OK;
}


/**
 * Function called when we're done processing the
 * HTTP /aml/$OFFICER_PUB/legitimizations GET request.
 *
 * @param cls the `struct TALER_EXCHANGE_AmlLegitimizationsGetHandle`
 * @param response_code HTTP response code, 0 on error
 * @param response parsed JSON result, NULL on error
 */
static void
handle_aml_legitimizations_get_finished (void *cls,
                                         long response_code,
                                         const void *response)
{
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh = cls;
  const json_t *j = response;
  struct TALER_EXCHANGE_AmlLegitimizationsGetResult result = {
    .hr.reply = j,
    .hr.http_status = (unsigned int) response_code
  };

  algh->job = NULL;
  switch (response_code)
  {
  case 0:
    result.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    break;
  case MHD_HTTP_OK:
    if (GNUNET_OK !=
        handle_aml_legitimizations_get_ok (algh,
                                           j))
    {
      result.hr.http_status = 0;
      result.hr.ec = TALER_EC_GENERIC_REPLY_MALFORMED;
    }
    break;
  case MHD_HTTP_NO_CONTENT:
    /* can happen */
    break;
  case MHD_HTTP_BAD_REQUEST:
    /* This should never happen, either us or the exchange is buggy
       (or API version conflict); just pass JSON reply to the application */
    result.hr.ec = TALER_JSON_get_error_code (j);
    result.hr.hint = TALER_JSON_get_error_hint (j);
    break;
  case MHD_HTTP_UNAUTHORIZED:
    /* Invalid officer credentials */
    result.hr.ec = TALER_JSON_get_error_code (j);
    result.hr.hint = TALER_JSON_get_error_hint (j);
    break;
  case MHD_HTTP_FORBIDDEN:
    /* Officer not authorized for this operation */
    result.hr.ec = TALER_JSON_get_error_code (j);
    result.hr.hint = TALER_JSON_get_error_hint (j);
    break;
  case MHD_HTTP_NOT_FOUND:
    /* Officer not found */
    result.hr.ec = TALER_JSON_get_error_code (j);
    result.hr.hint = TALER_JSON_get_error_hint (j);
    break;
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    /* Server had an internal issue; we should retry, but this API
       leaves this to the application */
    result.hr.ec = TALER_JSON_get_error_code (j);
    result.hr.hint = TALER_JSON_get_error_hint (j);
    break;
  default:
    /* unexpected response code */
    GNUNET_break_op (0);
    result.hr.ec = TALER_JSON_get_error_code (j);
    result.hr.hint = TALER_JSON_get_error_hint (j);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d for GET %s\n",
                (unsigned int) response_code,
                (int) result.hr.ec,
                algh->url);
    break;
  }
  if (NULL != algh->cb)
  {
    algh->cb (algh->cb_cls,
              &result);
    algh->cb = NULL;
  }
  TALER_EXCHANGE_aml_legitimizations_get_cancel (algh);
}


struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *
TALER_EXCHANGE_aml_legitimizations_get_create (
  struct GNUNET_CURL_Context *ctx,
  const char *exchange_base_url,
  const struct TALER_AmlOfficerPrivateKeyP *officer_priv)
{
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh;

  algh = GNUNET_new (struct TALER_EXCHANGE_AmlLegitimizationsGetHandle);
  algh->ctx = ctx;
  algh->exchange_base_url = GNUNET_strdup (exchange_base_url);
  GNUNET_CRYPTO_eddsa_key_get_public (&officer_priv->eddsa_priv,
                                      &algh->officer_pub.eddsa_pub);
  TALER_officer_aml_query_sign (officer_priv,
                                &algh->officer_sig);
  algh->options.limit = -20; /* Default to last 20 entries */
  algh->options.offset = INT64_MAX; /* Default to maximum row id */
  algh->options.active = TALER_EXCHANGE_YNA_ALL; /* Default to all */
  return algh;
}


enum GNUNET_GenericReturnValue
TALER_EXCHANGE_aml_legitimizations_get_set_options_ (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh,
  unsigned int num_options,
  const struct TALER_EXCHANGE_AmlLegitimizationsGetOptionValue *options)
{
  for (unsigned int i = 0; i < num_options; i++)
  {
    switch (options[i].option)
    {
    case TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_END:
      return GNUNET_OK;
    case TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_LIMIT:
      algh->options.limit = options[i].details.limit;
      break;
    case TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_OFFSET:
      algh->options.offset = options[i].details.offset;
      break;
    case TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_H_PAYTO:
      algh->options.h_payto = options[i].details.h_payto;
      break;
    case TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_OPTION_ACTIVE:
      algh->options.active = options[i].details.active;
      break;
    default:
      GNUNET_break (0);
      return GNUNET_NO;
    }
  }
  return GNUNET_OK;
}


enum TALER_EXCHANGE_AmlLegitimizationsGetStartError
TALER_EXCHANGE_aml_legitimizations_get_start (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh,
  TALER_EXCHANGE_AmlLegitimizationsGetCallback cb,
  TALER_EXCHANGE__AML_LEGITIMIZATIONS_GET_RESULT_CLOSURE *cb_cls)
{
  char officer_pub_str[sizeof (struct TALER_AmlOfficerPublicKeyP) * 2];
  char arg_str[sizeof (officer_pub_str) + 64];
  char limit_str[24];
  char offset_str[24];
  char paytoh_str[sizeof (struct TALER_NormalizedPaytoHashP) * 2];

  if (NULL != algh->job)
  {
    GNUNET_break (0);
    return TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_E_AGAIN;
  }
  algh->cb = cb;
  algh->cb_cls = cb_cls;
  if (algh->options.offset > INT64_MAX)
  {
    GNUNET_break (0);
    algh->options.offset = INT64_MAX;
  }
  {
    char *end;

    end = GNUNET_STRINGS_data_to_string (
      &algh->officer_pub,
      sizeof (algh->officer_pub),
      officer_pub_str,
      sizeof (officer_pub_str));
    *end = '\0';
  }
  if (NULL != algh->options.h_payto)
  {
    char *end;

    end = GNUNET_STRINGS_data_to_string (
      algh->options.h_payto,
      sizeof (struct TALER_NormalizedPaytoHashP),
      paytoh_str,
      sizeof (paytoh_str));
    *end = '\0';
  }
  /* Build query parameters */
  GNUNET_snprintf (offset_str,
                   sizeof (offset_str),
                   "%llu",
                   (unsigned long long) algh->options.offset);
  GNUNET_snprintf (limit_str,
                   sizeof (limit_str),
                   "%lld",
                   (long long) algh->options.limit);
  GNUNET_snprintf (arg_str,
                   sizeof (arg_str),
                   "aml/%s/legitimizations",
                   officer_pub_str);
  algh->url = TALER_url_join (algh->exchange_base_url,
                              arg_str,
                              "limit",
                              limit_str,
                              "offset",
                              ( (algh->options.limit > 0) &&
                                (0 == algh->options.offset) ) ||
                              ( (algh->options.limit <= 0) &&
                                (INT64_MAX <= algh->options.offset) )
                              ? NULL
                              : offset_str,
                              "h_payto",
                              NULL == algh->options.h_payto
                              ? NULL
                              : paytoh_str,
                              "active",
                              TALER_EXCHANGE_YNA_ALL == algh->options.active
                              ? NULL
                              : TALER_yna_to_string (algh->options.active),
                              NULL);
  if (NULL == algh->url)
  {
    GNUNET_break (0);
    return TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_E_INTERNAL;
  }

  {
    CURL *eh;

    eh = TALER_EXCHANGE_curl_easy_get_ (algh->url);
    if (NULL == eh)
    {
      GNUNET_break (0);
      GNUNET_free (algh->url);
      algh->url = NULL;
      return TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_E_INTERNAL;
    }

    /* Add authentication header for AML officer */
    {
      char *hdr;
      char sig_str[sizeof (algh->officer_sig) * 2];
      char *end;

      end = GNUNET_STRINGS_data_to_string (
        &algh->officer_sig,
        sizeof (algh->officer_sig),
        sig_str,
        sizeof (sig_str));
      *end = '\0';
      GNUNET_asprintf (&hdr,
                       "%s: %s",
                       TALER_AML_OFFICER_SIGNATURE_HEADER,
                       sig_str);
      algh->job_headers = curl_slist_append (NULL,
                                             hdr);
      GNUNET_free (hdr);
    }
    algh->job
      = GNUNET_CURL_job_add2 (
          algh->ctx,
          eh,
          algh->job_headers,
          &handle_aml_legitimizations_get_finished,
          algh);
  }
  return TALER_EXCHANGE_AML_LEGITIMIZATIONS_GET_START_OK;
}


void
TALER_EXCHANGE_aml_legitimizations_get_cancel (
  struct TALER_EXCHANGE_AmlLegitimizationsGetHandle *algh)
{
  if (NULL != algh->job)
  {
    GNUNET_CURL_job_cancel (algh->job);
    algh->job = NULL;
  }
  curl_slist_free_all (algh->job_headers);
  GNUNET_free (algh->exchange_base_url);
  GNUNET_free (algh->url);
  GNUNET_free (algh);
}


/* end of exchange_api_aml_legitimizations_get.c */
