/*
 * This file is part of LibEuFin.
 * Copyright (C) 2023-2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.testing.test
import tech.libeufin.common.crypto.CryptoUtil
import tech.libeufin.common.asUtf8
import tech.libeufin.nexus.*
import tech.libeufin.nexus.cli.LibeufinNexus
import java.io.ByteArrayOutputStream
import java.io.PrintStream
import kotlin.io.path.*
import kotlin.test.Test
import kotlin.test.assertEquals

val nexusCmd = LibeufinNexus()

fun CliktCommand.testErr(cmd: String, msg: String) {
    val prevOut = System.err
    val tmpOut = ByteArrayOutputStream()
    System.setErr(PrintStream(tmpOut))
    val result = test(cmd)
    System.setErr(prevOut)
    val tmpStr = tmpOut.asUtf8()
    println(tmpStr)
    assertEquals(1, result.statusCode, "'$cmd' should have failed")
    val line = tmpStr.substringAfterLast(" - ").trimEnd('\n')
    println(line)
    assertEquals(msg, line)
}

class CliTest {
    /** Test error format related to the keying process */
    @Test
    fun keys() {
        val cmds = listOf("ebics-submit", "ebics-fetch")
        val allCmds = listOf("ebics-submit", "ebics-fetch", "ebics-setup")
        val conf = "conf/test.conf"
        val nexusCfg = nexusConfig(Path(conf))
        val cfg = nexusCfg.ebics
        val clientKeysPath = cfg.clientPrivateKeysPath
        val bankKeysPath = cfg.bankPublicKeysPath
        clientKeysPath.parent!!.createParentDirectories()
        clientKeysPath.parent!!.toFile().setWritable(true)
        bankKeysPath.parent!!.createDirectories()
        
        // Missing client keys
        clientKeysPath.deleteIfExists()
        for (cmd in cmds) {
            nexusCmd.testErr("$cmd -c $conf", "Missing client private keys file at '$clientKeysPath', run 'libeufin-nexus ebics-setup' first")
        }
        // Empty client file
        clientKeysPath.createFile()
        for (cmd in allCmds) {
            nexusCmd.testErr("$cmd -c $conf", "Could not decode client private keys at '$clientKeysPath': Expected start of the object '{', but had 'EOF' instead at path: $\nJSON input: ")
        }
        // Bad client json
        clientKeysPath.writeText("CORRUPTION", Charsets.UTF_8)
        for (cmd in allCmds) {
            nexusCmd.testErr("$cmd -c $conf", "Could not decode client private keys at '$clientKeysPath': Unexpected JSON token at offset 0: Expected start of the object '{', but had 'C' instead at path: $\nJSON input: CORRUPTION")
        }
        // Missing permission
        clientKeysPath.toFile().setReadable(false)
        if (!clientKeysPath.isReadable()) { // Skip if root
            for (cmd in allCmds) {
                nexusCmd.testErr("$cmd -c $conf", "Could not read client private keys at '$clientKeysPath': permission denied")
            }
        }
        // Unfinished client
        persistClientKeys(generateNewKeys(), clientKeysPath)
        for (cmd in cmds) {
            nexusCmd.testErr("$cmd -c $conf", "Unsubmitted client private keys, run 'libeufin-nexus ebics-setup' first")
        }

        // Missing bank keys
        persistClientKeys(generateNewKeys().apply {
            submitted_hia = true
            submitted_ini = true
        }, clientKeysPath)
        bankKeysPath.deleteIfExists()
        for (cmd in cmds) {
            nexusCmd.testErr("$cmd -c $conf", "Missing bank public keys at '$bankKeysPath', run 'libeufin-nexus ebics-setup' first")
        }
        // Empty bank file
        bankKeysPath.createFile()
        for (cmd in allCmds) {
            nexusCmd.testErr("$cmd -c $conf", "Could not decode bank public keys at '$bankKeysPath': Expected start of the object '{', but had 'EOF' instead at path: $\nJSON input: ")
        }
        // Bad bank json
        bankKeysPath.writeText("CORRUPTION", Charsets.UTF_8)
        for (cmd in allCmds) {
            nexusCmd.testErr("$cmd -c $conf", "Could not decode bank public keys at '$bankKeysPath': Unexpected JSON token at offset 0: Expected start of the object '{', but had 'C' instead at path: $\nJSON input: CORRUPTION")
        }
        // Missing permission
        bankKeysPath.toFile().setReadable(false)
        if (!bankKeysPath.isReadable()) { // Skip if root
            for (cmd in allCmds) {
                nexusCmd.testErr("$cmd -c $conf", "Could not read bank public keys at '$bankKeysPath': permission denied")
            }
        }
        // Unfinished bank
        persistBankKeys(BankPublicKeysFile(
            bank_authentication_public_key = CryptoUtil.genRSAPublic(2048),
            bank_encryption_public_key = CryptoUtil.genRSAPublic(2048),
            accepted = false
        ), bankKeysPath)
        for (cmd in cmds) {
            nexusCmd.testErr("$cmd -c $conf", "Unaccepted bank public keys, run 'libeufin-nexus ebics-setup' until accepting the bank keys")
        }

        // Missing permission
        clientKeysPath.deleteIfExists()
        clientKeysPath.parent!!.toFile().setWritable(false)
        if (!clientKeysPath.parent!!.isWritable()) { // Skip if root
            nexusCmd.testErr("ebics-setup -c $conf", "Could not write client private keys at '$clientKeysPath': permission denied on '${clientKeysPath.parent}'")
        }
    }

    /** Test server check */
    @Test
    fun serveCheck() {
        val confs = listOf(
            "mini" to 1,
            "test" to 0
        )
        for ((conf, statusCode) in confs) {
            val result = nexusCmd.test("serve --check -c conf/$conf.conf")
            assertEquals(statusCode, result.statusCode)
        }
    }

    /** Test list cmds */
    @Test
    fun listCheck() = setup { db, _ ->
        fun check() {
            for (list in listOf("incoming", "outgoing", "initiated", "initiated --awaiting-ack")) {
                val result = nexusCmd.test("list $list -c conf/test.conf")
                assertEquals(0, result.statusCode)
            }
        }
        // Check empty
        check()
        // Check with transactions
        registerIn(db)
        registerOut(db)
        check()
        // Check with taler transactions
        talerableOut(db)
        talerableIn(db)
        talerableKycIn(db)
        check()
        // Check with incomplete
        registerIncompleteIn(db)
        talerableIncompleteIn(db)
        registerIncompleteOut(db)
        check()
    }
}